
}

// =================================================================================
// 1. LÓGICA DE CONTROLE DE SESSÃO E INICIALIZAÇÃO DE VARIÁVEIS DE IDENTIFICAÇÃO
// =================================================================================
$usuario_id = (int)($_SESSION['usuario']['id'] ?? 0);
$perfil = $_SESSION['usuario']['perfil'] ?? '';

$is_professor = strtoupper(trim($perfil)) === 'PROFESSOR';
$is_simulating = $_SESSION['simulando'] ?? false; 

// Para fins de consulta, se estiver simulando, use o ID do gestor ou continue usando o do professor
// Nota: Seu código original usa $usuario_id, mantendo a lógica de que o gestor opera como o professor.
$id_para_consulta = $usuario_id; 

if (!$is_professor && !$is_simulating) {
    header('Location: login.php');
    exit;
}
if ($usuario_id <= 0) {
    header('Location: login.php');
    exit;
}

// Nome do professor alvo
$nome_professor = '';
$sqlNome = "SELECT nome FROM usuarios WHERE id = ? LIMIT 1";
$stmt = $conn->prepare($sqlNome);
if ($stmt) {
    $stmt->bind_param("i", $usuario_id);
    $stmt->execute();
    $resNome = $stmt->get_result();
    if ($resNome && $resNome->num_rows === 1) {
        $rowNome = $resNome->fetch_assoc();
        $nome_professor = $rowNome['nome'];
    }
    $stmt->close();
}

// Ano lectivo e Variáveis Estáticas
$ano_lectivo_atual = '2025/2026';
$nome_republica = "REPÚBLICA DE ANGOLA";
$nome_governo = "GOVERNO PROVINCIAL DE BENGUELA";
$nome_gabinete = "GABINETE PROVINCIAL DA EDUCAÇÃO";
$nome_escola = "COMPLEXO ESCOLAR CATÓLICO Nº 0026 BLA - PADRE EMIL FRICK - CATUMBELA";
$nome_sub_diretor = "";
$localidade = "CATUMBELA";

// Variáveis dinâmicas da turma (inicializadas)
$classe_dinamica = '';
$nome_turma_dinamica = '';
$curso_area_dinamica = '';
$turno_dinamico = '';
$sala_dinamica = '';
$turma_completa = '';
$nome_disciplina = '';
$nome_turma_selecionada = '';
$prazo_fechado = false;
$alunos = [];
$notas_por_aluno = [];
$mac_por_aluno = [];

// =================================================================================
// 2. FUNÇÃO BUSCAR MAC
// =================================================================================
function buscarMacConsolidado($conn, $aluno_id, $disciplina_id, $turma_id, $trimestre) {
    // Busca o MAC do aluno para o trimestre selecionado
    $sql = "SELECT mac FROM avaliacoes_continuas
            WHERE aluno_id = ? AND disciplina_id = ? AND turma_id = ? AND trimestre = ?";
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        error_log("Erro de preparação SQL (MAC): " . $conn->error);
        return 0.0;
    }
    
    // Bind: 4 inteiros (IDs e trimestre)
    $stmt->bind_param("iiii", $aluno_id, $disciplina_id, $turma_id, $trimestre);
    $stmt->execute();
    $result = $stmt->get_result();
    $stmt->close();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        // Garante que o valor retornado é float, mesmo que seja null no DB (retornando 0.0)
        return ($row['mac'] !== null) ? (float)$row['mac'] : 0.0; 
    }
    return 0.0;
}

// =================================================================================
// 3. INICIALIZAÇÃO DE VARIÁVEIS DE FILTRO (LÊ GET) E CARREGAMENTO DE LISTAS
// =================================================================================
$turma_id = (int)($_GET['turma_id'] ?? 0);
$disciplina_id = (int)($_GET['disciplina_id'] ?? 0);
$trimestre_selecionado = (int)($_GET['trimestre'] ?? 0);

// Busca de Turmas (Carrega para o SELECT)
$turmas = [];
$sqlTurmas = "
    SELECT DISTINCT t.id, t.nome, t.classe, t.sala, t.curso, t.turno
    FROM turmas t
    INNER JOIN atribuicoes a ON t.id = a.turma_id
    WHERE a.professor_id = ?
    ORDER BY t.classe, t.nome
";
$stmt = $conn->prepare($sqlTurmas);
$stmt->bind_param("i", $id_para_consulta);
$stmt->execute();
$turmas_res = $stmt->get_result();
$turmas = [];
while ($row = $turmas_res->fetch_assoc()) {
    $partes = [];
    if ($row['classe']) $partes[] = $row['classe'];
    if ($row['nome']) $partes[] = $row['nome'];
    if ($row['curso']) $partes[] = $row['curso'];
    if ($row['turno']) $partes[] = $row['turno'];
    if ($row['sala']) $partes[] = $row['sala'];
    $row['descricao_formatada'] = implode('-', $partes);
    $turmas[] = $row;
}
$stmt->close();

// Busca de Disciplinas (Carrega para o SELECT, dependente de turma_id)
$disciplinas = [];
if ($turma_id > 0) {
    $sqlDisciplinas = "
        SELECT DISTINCT d.id, d.nome
        FROM atribuicoes a
        INNER JOIN disciplinas d ON a.disciplina_id = d.id
        WHERE a.professor_id = ? AND a.turma_id = ?
        ORDER BY d.nome
    ";
    $stmt = $conn->prepare($sqlDisciplinas);
    $stmt->bind_param("ii", $id_para_consulta, $turma_id);
    $stmt->execute();
    $disciplinas_res = $stmt->get_result();
    while ($disciplina = $disciplinas_res->fetch_assoc()) {
        $disciplinas[] = $disciplina;
    }
    $stmt->close();
}

// Mensagens de feedback (Usa SESSION para garantir que a mensagem não suma após o redirecionamento)
$mensagem_sucesso = $_SESSION['mensagem_sucesso'] ?? '';
$mensagem_erro = $_SESSION['mensagem_erro'] ?? '';
unset($_SESSION['mensagem_sucesso'], $_SESSION['mensagem_erro']);


// =================================================================================
// 4. LÓGICA DE SALVAR NOTAS (INTEGRADA E ROBUSTA)
// Garante o salvamento e o redirecionamento para reexibir os dados atualizados.
// =================================================================================
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['salvar_notas'])) { // Adicionado check de 'salvar_notas'
    
    // O professor SÓ pode salvar se o prazo não estiver bloqueado, OU se o gestor estiver simulando.
    $pode_salvar = $is_simulating || ($is_professor && $turma_id > 0 && $disciplina_id > 0 && $trimestre_selecionado > 0); 

    if (!$pode_salvar) {
        $_SESSION['mensagem_erro'] = "Acesso negado para salvar notas. Você não tem permissão ou o prazo expirou.";
        header('Location: mini_pauta_professor1.php');
        exit;
    }

    // Receber dados do formulário (Usando POST para consistência)
    $aluno_ids          = $_POST['aluno_id'] ?? [];
    $disciplina_id_post = (int)($_POST['disciplina_id'] ?? 0);
    $turma_id_post      = (int)($_POST['turma_id'] ?? 0);
    $trimestre_post     = (int)($_POST['trimestre'] ?? 0);
    $ano_lectivo        = $ano_lectivo_atual;

    if (empty($aluno_ids) || $disciplina_id_post <= 0 || $turma_id_post <= 0 || $trimestre_post <= 0) {
        $_SESSION['mensagem_erro'] = "Erro ao salvar: dados incompletos.";
        // Redireciona com os filtros para reexibir a pauta
        header("Location: mini_pauta_professor1.php?turma_id=$turma_id_post&disciplina_id=$disciplina_id_post&trimestre=$trimestre_post");
        exit;
    }

    // Iniciar Transação
    $conn->begin_transaction();

    try {
        
        // Buscar Classe da Turma (Usando $turma_id_post)
        $classe = "";
        $stmtC = $conn->prepare("SELECT classe FROM turmas WHERE id=?");
        $stmtC->bind_param("i", $turma_id_post);
        $stmtC->execute();
        $rC = $stmtC->get_result();
        if ($rC->num_rows > 0) {
            $classe = $rC->fetch_assoc()['classe'];
        }
        $stmtC->close();

        // Preparar UPSERT (REPLACE INTO)
        $sql = "
            REPLACE INTO notas (
                aluno_id, turma_id, disciplina_id, ano_lectivo, trimestre, classe,
                mac, npp, npt, mt,
                mt1, mt2, mt3,
                mfd, mf, neo, nee, mec, exame, nota_final, observacao
            ) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
        ";

        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro preparar REPLACE INTO: " . $conn->error);
        }

        // Loop dos Alunos
        foreach ($aluno_ids as $aluno_id) {

            $aluno_id = (int)$aluno_id;

            // Coleta notas do formulário (MAC, NPP, NPT, NEO, NEE, OBS)
            // MAC: Usado apenas se for editável, senão, busca do avaliacoes_continuas
            $mac_input = str_replace(",", ".", $_POST["mac_{$aluno_id}"] ?? '0');
            $npp_input = str_replace(",", ".", $_POST["npp_{$trimestre_post}_{$aluno_id}"] ?? '0');
            $npt_input = str_replace(",", ".", $_POST["npt_{$trimestre_post}_{$aluno_id}"] ?? '0');
            
            $neo_input = str_replace(",", ".", $_POST["neo_{$aluno_id}"] ?? '0'); // Sem trimestre no nome do input
            $nee_input = str_replace(",", ".", $_POST["nee_{$aluno_id}"] ?? '0'); // Sem trimestre no nome do input
            
            $obs_input = $_POST["obs_{$aluno_id}"] ?? '';

            // Convertendo para float
            $mac_post = (float)$mac_input; // MAC postado (se editável)
            $npp = (float)$npp_input; 
            $npt = (float)$npt_input;
            $neo_input_float = (float)$neo_input;
            $nee_input_float = (float)$nee_input;

            // Variáveis de notas (carregadas ou inicializadas)
            $mac = 0.0; $mt = 0.0;
            $mt1 = 0.0; $mt2 = 0.0; $mt3 = 0.0;
            $mfd = $mf = $neo = $nee = $mec = $exame = $nota_final = 0.0;

            // --- BUSCAR DADOS EXISTENTES (Para consolidar MTs anteriores e Notas Finais) ---
            $stmtB = $conn->prepare("SELECT * FROM notas WHERE aluno_id=? AND disciplina_id=? AND turma_id=? AND ano_lectivo=?");
            $stmtB->bind_param("iiis", $aluno_id, $disciplina_id_post, $turma_id_post, $ano_lectivo);
            $stmtB->execute();
            $resB = $stmtB->get_result();

            $dados_existentes = [];
            while ($row = $resB->fetch_assoc()) {
                $dados_existentes[(int)$row['trimestre']] = $row;
            }
            $stmtB->close();

            // Recupera MTs e Notas Finais existentes de TODOS os trimestres
            $mt1 = (float)($dados_existentes[1]['mt'] ?? 0.0);
            $mt2 = (float)($dados_existentes[2]['mt'] ?? 0.0);
            $mt3 = (float)($dados_existentes[3]['mt'] ?? 0.0);
            
            // Recuperar finais existentes (só existem no registro do 3º trimestre)
            if (isset($dados_existentes[3])) {
                $mfd = (float)($dados_existentes[3]['mfd'] ?? 0.0);
                $mf =  (float)($dados_existentes[3]['mf'] ?? 0.0);
                $neo = (float)($dados_existentes[3]['neo'] ?? 0.0);
                $nee = (float)($dados_existentes[3]['nee'] ?? 0.0);
                $mec = (float)($dados_existentes[3]['mec'] ?? 0.0);
                $exame = (float)($dados_existentes[3]['exame'] ?? 0.0);
                $nota_final = (float)($dados_existentes[3]['nota_final'] ?? 0.0);
            }

            // ====================================================================
            // MAC: Prioridade 1: Valor postado (se gestor); Prioridade 2: avaliacoes_continuas
            // ====================================================================
            if ($is_simulating) {
                // Se estiver simulando, usa o valor postado do MAC
                $mac = $mac_post;
            } else {
                // Senão, usa o MAC consolidado da tabela de avaliação contínua
                $mac = buscarMacConsolidado($conn, $aluno_id, $disciplina_id_post, $turma_id_post, $trimestre_post);
            }

            // ====================================================================
            // ⭐️ CÁLCULO DA MÉDIA TRIMESTRAL (MT) - MT = (MAC + NPP + NPT) / 3
            // ====================================================================
            $mt = ($mac + $npp + $npt) / 3.0;
            $mt = round($mt, 2); 

            // Atualiza o MT do trimestre atual com o valor CALCULADO
            if ($trimestre_post == 1) $mt1 = $mt;
            if ($trimestre_post == 2) $mt2 = $mt;
            if ($trimestre_post == 3) $mt3 = $mt;

            // Atualizar NEO e NEE (somente no 3º trimestre) com os valores postados
            if ($trimestre_post == 3) {
                $neo = $neo_input_float;
                $nee = $nee_input_float;
            }

            $mt_principal = $mt;
            $npp_db_value = $npp;
            $npt_db_value = $npt; // Usando $npt como float

            // ================== RECALCULAR MÉDIAS NO 3º TRIMESTRE ==================
            if ($trimestre_post == 3) {
                // 1. Calcular MFD
                $mts_validos = [];
                if ($mt1 > 0) $mts_validos[] = $mt1;
                if ($mt2 > 0) $mts_validos[] = $mt2;
                if ($mt3 > 0) $mts_validos[] = $mt3;

                if (count($mts_validos) === 3) {
                    $mfd = array_sum($mts_validos) / 3;
                } else {
                    $mfd = 0;
                }
                $mfd = round($mfd, 2);

                // 2. Calcular MEC
                $mec = ($neo > 0 && $nee > 0) ? (($neo + $nee) / 2) : 0;
                $mec = round($mec, 2);

                // 3. Calcular MF
                if ($mfd > 0) {
                    if ($mec > 0) {
                        $mf = ($mfd * 0.6) + ($mec * 0.4);
                    } else if ($neo > 0 || $nee > 0) {
                        // Se só tem uma nota de recurso
                        $nota_exame = $neo > 0 ? $neo : $nee;
                         $mf = ($mfd * 0.6) + ($nota_exame * 0.4);
                    } else {
                        $mf = $mfd;
                    }
                } else {
                    $mf = 0;
                }
                $mf = round($mf, 2);

                // 4. Nota final (arredondamento)
                if ($mf >= 9.5 && $mf < 10) {
                    $nota_final = 10.0;
                } else {
                    $nota_final = $mf;
                }
            }
            // ===================================================================================

            // --------------------------------------------------------------------
            // EXECUÇÃO DO REPLACE INTO
            // --------------------------------------------------------------------
            // OBS: Adicionada a variável $obs_input (string) no final do bind_param
            $stmt->bind_param(
                "iisissdddddddddddddds",
                $aluno_id, $turma_id_post, $disciplina_id_post, $ano_lectivo, $trimestre_post, $classe,
                $mac, $npp_db_value, $npt_db_value, $mt_principal, // MAC, NPP, NPT, MT (principal)
                $mt1, $mt2, $mt3, // MT1, MT2, MT3 (consolidados)
                $mfd, $mf, $neo, $nee, $mec, $exame, $nota_final, // Finais
                $obs_input // Nova coluna de Observação (string)
            );

            if (!$stmt->execute()) {
                throw new Exception("Erro ao executar REPLACE INTO para Aluno ID {$aluno_id}: " . $stmt->error);
            }

        } // Fim do loop foreach ($aluno_ids as $aluno_id)


        // 5. FINALIZAÇÃO DA TRANSAÇÃO
        $conn->commit();
        $_SESSION['mensagem_sucesso'] = "Notas salvas e consolidadas com sucesso! Os campos MAC, NPP, NPT, MT e Notas Finais foram atualizados.";
        // Redireciona com os filtros para reexibir a pauta
        header("Location: mini_pauta_professor1.php?turma_id=$turma_id_post&disciplina_id=$disciplina_id_post&trimestre=$trimestre_post");
        exit;

    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['mensagem_erro'] = "Erro ao salvar notas: " . $e->getMessage();
        // Redireciona com os filtros para reexibir a pauta
        header("Location: mini_pauta_professor1.php?turma_id=$turma_id_post&disciplina_id=$disciplina_id_post&trimestre=$trimestre_post");
        exit;
    }
}
// Fim da Lógica de Salvar Notas
// =================================================================================


// =================================================================================
// 6. LÓGICA DE BLOQUEIO DE EDIÇÃO (PRAZO)
// Isso define as variáveis $mac_readonly, $npp_readonly e $npt_readonly
// =================================================================================
$data_hoje = date("Y-m-d");

// Variáveis de Estado (Inicia como liberado)
$npt_bloqueado = false;
$npp_bloqueado = false;
$motivo_bloqueio_npt = "";
$motivo_bloqueio_npp = "";

if ($trimestre_selecionado > 0) { // Só verifica se um trimestre foi selecionado

    // 1. VERIFICAR PRAZO DO NPT
    // -------------------------
    $tipo_nota = 'NPT';
    $stmt_npt = $conn->prepare("
        SELECT data_inicio, data_fim
        FROM prazos_lancamento
        WHERE ano_lectivo = ? AND trimestre = ? AND tipo_nota = ?
        LIMIT 1
    ");
    $stmt_npt->bind_param("sis", $ano_lectivo_atual, $trimestre_selecionado, $tipo_nota);
    $stmt_npt->execute();
    $resultado_npt = $stmt_npt->get_result()->fetch_assoc();

    if($resultado_npt) {
        $inicio = $resultado_npt['data_inicio'];
        $fim = $resultado_npt['data_fim'];

        if ($data_hoje < $inicio) {
            $npt_bloqueado = true;
            $motivo_bloqueio_npt = "O prazo para NPT só inicia em " . date("d/m/Y", strtotime($inicio));
        } elseif ($data_hoje > $fim) {
            $npt_bloqueado = true;
            $motivo_bloqueio_npt = "O prazo para NPT terminou no dia " . date("d/m/Y", strtotime($fim));
        }
    } else {
        $npt_bloqueado = true;
        $motivo_bloqueio_npt = "O prazo para NPT ainda não foi definido.";
    }
    $stmt_npt->close();


    // 2. VERIFICAR PRAZO DO NPP
    // -------------------------
    $tipo_nota = 'NPP';
    $stmt_npp = $conn->prepare("
        SELECT data_inicio, data_fim
        FROM prazos_lancamento
        WHERE ano_lectivo = ? AND trimestre = ? AND tipo_nota = ?
        LIMIT 1
    ");
    $stmt_npp->bind_param("sis", $ano_lectivo_atual, $trimestre_selecionado, $tipo_nota);
    $stmt_npp->execute();
    $resultado_npp = $stmt_npp->get_result()->fetch_assoc();

    if ($resultado_npp) {
        $inicio = $resultado_npp['data_inicio'];
        $fim = $resultado_npp['data_fim'];

        if ($data_hoje < $inicio) {
            $npp_bloqueado = true;
            $motivo_bloqueio_npp = "O prazo para NPP só inicia em " . date("d/m/Y", strtotime($inicio));
        } elseif ($data_hoje > $fim) {
            $npp_bloqueado = true;
            $motivo_bloqueio_npp = "O prazo para NPP terminou no dia " . date("d/m/Y", strtotime($fim));
        }
    } else {
        $npp_bloqueado = true;
        $motivo_bloqueio_npp = "O prazo para NPP ainda não foi definido.";
    }
    $stmt_npp->close();

} else {
    // Se nenhum trimestre foi selecionado, bloqueia por padrão
    $npt_bloqueado = true;
    $npp_bloqueado = true;
}


// --- DEFINIÇÃO DAS VARIÁVEIS DE CONTROLE DE EDIÇÃO ---
// MAC só é editável em simulação
$mac_editavel = $is_simulating;
// NPP/NPT são editáveis se o prazo estiver aberto OU se estiver em simulação
$npp_editavel = !$npp_bloqueado || $is_simulating;
$npt_editavel = !$npt_bloqueado || $is_simulating;

// Define o atributo readonly e estilo para o MAC
$mac_readonly = $mac_editavel ? '' : 'readonly';
$mac_style = $mac_editavel ? 'background:#fff;' : 'background:#eee;';

// Define o atributo readonly e estilo para NPP e NPT
$npp_readonly = $npp_editavel ? '' : 'readonly';
$npp_style = $npp_editavel ? 'background:#fff;' : 'background:#eee;';

$npt_readonly = $npt_editavel ? '' : 'readonly';
$npt_style = $npt_editavel ? 'background:#fff;' : 'background:#eee;';
// =================================================================================


// =================================================================================
// 7. DETALHES DA TURMA E CARREGAMENTO DE ALUNOS E NOTAS PARA EXIBIÇÃO
// =================================================================================

// Detalhes da Turma para o Cabeçalho (Usa $turma_id vindo do GET)
if ($turma_id > 0) {
    foreach ($turmas as $turma) {
        if ($turma['id'] == $turma_id) {
            $nome_turma_selecionada = $turma['descricao_formatada'];
            
            $partes_turma = explode('-', $nome_turma_selecionada);
            $classe_dinamica = $partes_turma[0] ?? '';
            $nome_turma_dinamica = $partes_turma[1] ?? '';
            $curso_area_dinamica = $partes_turma[2] ?? '';
            $turno_dinamico = $partes_turma[3] ?? '';
            $sala_dinamica = $partes_turma[4] ?? '';

            $turma_completa = "Turma: " . $nome_turma_dinamica . " - " . $turno_dinamico . " sala nº " . $sala_dinamica;
            
            break;
        }
    }
}

// Busca de nome da Disciplina
if ($disciplina_id > 0) {
    $sqlNomeDisciplina = "SELECT nome FROM disciplinas WHERE id = ?";
    $stmt = $conn->prepare($sqlNomeDisciplina);
    $stmt->bind_param("i", $disciplina_id);
    $stmt->execute();
    $res = $stmt->get_result();
    if ($res->num_rows > 0) {
        $row = $res->fetch_assoc();
        $nome_disciplina = $row['nome'];
    }
    $stmt->close();
}

// Carrega Alunos e Notas para a Mini-Pauta
if ($turma_id > 0 && $disciplina_id > 0 && $trimestre_selecionado > 0) {
    // Verifica se o trimestre já foi fechado
    $sqlCheckFecho = "SELECT COUNT(*)
        FROM fecho_trimestre
        WHERE turma_id = ? AND disciplina_id = ? AND trimestre = ?";
    $stmtCheckFecho = $conn->prepare($sqlCheckFecho);
    $stmtCheckFecho->bind_param("iii", $turma_id, $disciplina_id, $trimestre_selecionado);
    $stmtCheckFecho->execute();
    $count = $stmtCheckFecho->get_result()->fetch_row()[0];
    $stmtCheckFecho->close();
    $prazo_fechado = ($count > 0);

    // Buscar alunos da turma
    $sqlAlunos = "SELECT id, nome FROM alunos WHERE turma_id = ? ORDER BY nome";
    $stmtAlunos = $conn->prepare($sqlAlunos);
    $stmtAlunos->bind_param("i", $turma_id);
    $stmtAlunos->execute();
    $alunos_res = $stmtAlunos->get_result();
    $alunos_data = $alunos_res->fetch_all(MYSQLI_ASSOC);
    $stmtAlunos->close();

    $alunos = $alunos_data;

    // Buscar notas já lançadas
    $aluno_ids_data = array_column($alunos_data, 'id');
    if (!empty($aluno_ids_data)) {
        
        $in_clause = str_repeat('?,', count($aluno_ids_data) - 1) . '?';

        // -------------------------------------------------------------
        // 1. BUSCA MAC DA TABELA AVALIACOES_CONTINUAS (Recente)
        // -------------------------------------------------------------
        $sqlMAC = "
            SELECT aluno_id, trimestre, mac
            FROM avaliacoes_continuas
            WHERE aluno_id IN ($in_clause)
            AND disciplina_id = ?
            AND turma_id = ?
        ";
        $params_types_mac = str_repeat('i', count($aluno_ids_data)) . 'ii';
        $params_mac = array_merge($aluno_ids_data, [$disciplina_id, $turma_id]);

        $stmtMAC = $conn->prepare($sqlMAC);
        call_user_func_array([$stmtMAC, 'bind_param'], refValues(array_merge([$params_types_mac], $params_mac)));
        $stmtMAC->execute();
        $resultMAC = $stmtMAC->get_result();
        
        while ($mac_row = $resultMAC->fetch_assoc()) {
            $mac_por_aluno[$mac_row['aluno_id']][$mac_row['trimestre']] = (float)$mac_row['mac'];
        }
        $stmtMAC->close();
        
        // -------------------------------------------------------------
        // 2. BUSCAR OUTRAS NOTAS (NPP, NPT, MT, etc.) DA TABELA NOTAS
        // -------------------------------------------------------------
        $sqlNotas = "
            SELECT * FROM notas
            WHERE aluno_id IN ($in_clause)
            AND disciplina_id = ?
            AND turma_id = ?
        ";
        $params_types = str_repeat('i', count($aluno_ids_data)) . 'ii';
        $params = array_merge($aluno_ids_data, [$disciplina_id, $turma_id]);

        $stmtNotas = $conn->prepare($sqlNotas);
        call_user_func_array([$stmtNotas, 'bind_param'], refValues(array_merge([$params_types], $params)));
        $stmtNotas->execute();
        $resultNotas = $stmtNotas->get_result();
        
        while ($nota = $resultNotas->fetch_assoc()) {
            $aluno = $nota['aluno_id'];
            $trimestre = $nota['trimestre'];
            
            // Unifica as notas da tabela 'notas'
            $notas_por_aluno[$aluno][$trimestre] = $nota;
            
            // SOBREPÕE o MAC (prioriza a tabela avaliacoes_continuas)
            if (isset($mac_por_aluno[$aluno][$trimestre])) {
                $notas_por_aluno[$aluno][$trimestre]['mac'] = $mac_por_aluno[$aluno][$trimestre];
            } else {
                 // Se não encontrou MAC em avaliacoes_continuas, garante que o MAC da tabela notas seja usado ou 0.0
                 $notas_por_aluno[$aluno][$trimestre]['mac'] = (float)($nota['mac'] ?? 0.0);
            }
            
            // -------------------------------------------------------------
            // ⭐️ LÓGICA REFORÇADA: RECALCULAR MT PARA EXIBIÇÃO (MT = (MAC + NPP + NPT) / 3)
            // -------------------------------------------------------------
            $mac_lido_exibicao = (float)($notas_por_aluno[$aluno][$trimestre]['mac'] ?? 0.0);
            $npp_lido_exibicao = (float)($notas_por_aluno[$aluno][$trimestre]['npp'] ?? 0.0);
            $npt_lido_exibicao = (float)($notas_por_aluno[$aluno][$trimestre]['npt'] ?? 0.0);
            
            // Recálculo usando os valores lidos (incluindo MAC da avaliacao_continua)
            if ($mac_lido_exibicao > 0.0 || $npp_lido_exibicao > 0.0 || $npt_lido_exibicao > 0.0) { 
                $mt_recalculado = ($mac_lido_exibicao + $npp_lido_exibicao + $npt_lido_exibicao) / 3.0;
                $notas_por_aluno[$aluno][$trimestre]['mt'] = round($mt_recalculado, 2);
            } else {
                $notas_por_aluno[$aluno][$trimestre]['mt'] = 0.0;
            }
            // -------------------------------------------------------------
        }
        $stmtNotas->close();
    }
}

// Variáveis injetadas no JavaScript
$notas_js = json_encode($notas_por_aluno);
$trimestre_js = (int)($trimestre_selecionado);
?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Lançar Notas - Mini Pauta</title>
    
    <link rel="stylesheet" href="css/all.min.css">
    <link rel="stylesheet" href="css/inter.css">
    <script src="js/tailwindcss.js"></script>
    <script src="js/chart.umd.min.js"></script>
    <style>
/* Estilos de interface (desktop/mobile) - Início */

.sidebar { transition: transform 0.3s; transform: translateX(-100%); z-index: 999; }
.sidebar.active { transform: translateX(0); }
.overlay { position: fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); z-index:998; display:none; }
.overlay.active { display:block; }

@media (min-width: 768px) {
    .sidebar { transform: translateX(0); }
    .content { margin-left: 16rem; }
    .overlay { display: none !important; }
}

input[type="number"]::-webkit-outer-spin-button,
input[type="number"]::-webkit-inner-spin-button { -webkit-appearance: none; margin: 0; }
input[type="number"] { -moz-appearance: textfield; }

/* Largura padrão dos inputs de nota na tela */
.nota-input {
    width: 50px; /* Mobile */
    font-size: 0.8rem;
    border-radius: 4px; 
    background-color: white; 
}
.mac-input {
     background-color: #f3f4f6; 
     font-weight: bold;
}

@media (min-width: 768px) {
    .nota-input {
        width: 80px; /* Desktop */
        font-size: 1rem;
    }
    /* Removida a regra de min-width para notas na tela */
}

/* Melhoria de Design para a Tabela (Desktop/Mobile) */
.print-table-container table {
    border-collapse: separate; 
    border-spacing: 0;
}
.print-table-container th, .print-table-container td {
    padding: 8px 10px;
    border-bottom: 1px solid #e5e7eb; 
    border-right: 1px solid #e5e7eb;
    text-align: center;
}
.print-table-container th:first-child, .print-table-container td:first-child {
    border-left: 1px solid #e5e7eb;
}
.print-table-container tr:last-child td {
    border-bottom: 1px solid #e5e7eb;
}
.print-table-container thead {
    background-color: #1e3a8a; 
    color: white;
}
.print-table-container tbody tr:hover {
    background-color: #f9f9f9;
}

/* CORREÇÕES ESPECÍFICAS PARA NOME DO ALUNO (TELA) */
.coluna-nome-aluno-tela {
    min-width: 180px; 
    text-align: left !important;
    /* Certificar que estas regras NÃO estejam ativas na tela se estiverem causando reticências: */
    /* white-space: nowrap; 
    overflow: hidden; 
    text-overflow: ellipsis; */
}

/* Células de Notas Calculadas (Estilo Padrão, será sobrescrito se tiver valor) */
.nota-calculada { background-color: #e0f2f1; font-weight: bold; color: #0f766e; } /* MT, MFD, MEC */
.nota-final-calc { background-color: #fef2f2; font-weight: bold; color: #b91c1c; } /* MF */

/* === ESTILOS DE NOTAS FINAIS === */
.nota-aprovada {
    background-color: #d1fae5 !important; /* Verde muito claro */
    color: #065f46 !important; /* Verde escuro */
    font-weight: bold;
}
.nota-reprovada {
    background-color: #fecaca !important; /* Vermelho muito claro */
    color: #991b1b !important; /* Vermelho escuro */
    font-weight: bold;
}

.estado-final-aprovado { background-color: #d1fae5; color: #065f46; font-weight: bold; }
.estado-final-reprovado { background-color: #fecaca; color: #991b1b; font-weight: bold; }
.estado-final-incompleto { background-color: #fef3c7; color: #b45309; font-weight: bold; }

/* Estilos de interface (desktop/mobile) - Fim */
/* ------------------------------------------- */


/* ================================================= */
/* ESTILOS DE IMPRESSÃO (A4) - OTIMIZAÇÃO MAXIMIZADA */
/* ================================================= */
@media print {
    @page {
        size: A4 portrait; 
        margin: 5mm; 
    }

    .print-hidden, .sidebar, header, nav, .overlay, .form-filtros, .btn-salvar-notas {
        display: none !important;
    }
    
    body, main, .content {
        margin: 0 !important;
        padding: 0 !important;
        min-height: initial !important;
        background-color: white !important;
    }

    .print-table-container {
        box-shadow: none !important;
        border: none !important;
    }
    table {
        width: 100% !important;
        border-collapse: collapse !important;
        font-size: 8pt; 
        table-layout: fixed; /* Essencial para que as larguras fixas funcionem */
    }
    th, td {
        border: 1px solid #000 !important;
        padding: 1px 3px !important; 
        color: #000 !important; 
        background-color: white !important; /* Branco na impressão */
        height: 15px; 
    }
    thead {
        -webkit-print-color-adjust: exact;
        color-adjust: exact;
        background-color: #d1d5db !important; 
        color: #000 !important;
    }
    
    /* Remove a cor da célula na impressão e mostra apenas o texto */
    .nota-aprovada, .nota-reprovada, .nota-calculada, .nota-final-calc {
        background-color: white !important;
        color: #000 !important;
    }

    /* --- AJUSTES DE LARGURA DAS COLUNAS (MAXIMIZAÇÃO DO NOME) --- */

    /* 1. COLUNA N.º DE ORDEM (Assumindo a 1ª coluna) */
    td:nth-child(1), th:nth-child(1) {
        width: 3% !important; 
        text-align: center !important;
    }

    /* 2. COLUNA DO NOME DO ALUNO (2ª COLUNA) - MAXIMIZADA E CORRIGIDA */
    .coluna-nome-aluno {
        width: 60% !important; /* MAXIMIZADO para 60% */
        min-width: 150px;
        text-align: left !important;
        padding-left: 5px !important;
        
        /* CORREÇÃO: Força linha única e elimina as reticências */
        white-space: nowrap; /* Garante que o nome fique em uma única linha */
        overflow: hidden; /* Oculta o que excede a largura (sem reticências) */
        text-overflow: clip; /* Garante que o corte não seja com reticências */
    }
    .coluna-nome-aluno-header {
        text-align: left !important;
        padding-left: 5px !important;
    } 


    /* Ocultar inputs e colunas de exame/mfd na impressão (Manter apenas o valor) */
    .nota-input { display: none !important; }

    /* CORREÇÃO/REFORÇO: Mostra o valor de nota a partir do data-nota-valor na impressão */
    .print-value::before { 
        content: attr(data-nota-valor); 
        display: block; 
        text-align: center; /* Centraliza o valor que aparece na impressão */
    }
    
    /* 3. COLUNAS DE NOTAS TRIMESTRAIS (MAC, NPT, MT - 3ª, 4ª, 5ª) - REDUZIDAS */
    td:nth-child(3), td:nth-child(4), td:nth-child(5) {
        width: 3% !important; /* REDUZIDO para 3% (Total 9%) */
    }
    
    /* 4. Colunas finais (MFD/NEO/NEE/MEC/MF/ESTADO - 6ª a 11ª) - REDUZIDAS */
    td:nth-child(6), td:nth-child(7), td:nth-child(8), td:nth-child(9), td:nth-child(10), td:nth-child(11) {
        width: 4% !important; /* REDUZIDO para 4% (Total 24%) */
    }
    
    /* 5. Coluna OBS. (12ª coluna) */
    td:nth-child(12), th:nth-child(12) {
        width: 4% !important; /* 4% restante */
    }
    
    /* --- CABEÇALHO --- */
    .print-header {
        display: block !important;
        text-align: center;
        margin-bottom: 5px;
    }
    .header-insignia img { width: 40px; height: auto; margin: 0 auto; display: block; }
    .header-text { line-height: 1.1; font-size: 8pt; margin-bottom: 3px; }
    .header-text p { margin: 1px 0; }
    .header-title { font-weight: bold; margin: 3px 0 3px 0; font-size: 9pt; }
    .header-info { text-align: left; font-size: 8pt; width: 100%; margin-top: 3px; display: table; }
    .header-info p { margin: 1px 0; display: table-row; }
    .header-info .info-label { font-weight: normal; white-space: nowrap; display: table-cell; padding-right: 5px; }
    .header-info .info-value { font-weight: bold; display: table-cell; }
    hr { margin: 5px 0 !important;}

    /* --- RODAPÉ DE ASSINATURAS --- */
    .print-footer {
        display: block !important;
        margin-top: 10px;
        width: 100%;
        font-size: 8pt;
    }
    .footer-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px; 
        width: 90%; 
        margin: 0 auto;
        text-align: center;
    }
    .assinatura-linha {
        margin-top: 20px;
        border-bottom: 1px solid #000;
        width: 100%;
        height: 1px;
    }
    .nome-assinado {
        margin-top: 2px;
        font-weight: bold;
    }
    .data-catumbela {
        margin-top: 5px;
        font-size: 8pt;
    }
    .data-catumbela span {
        border-bottom: 1px solid #000;
        padding: 0 5px; 
    }
}
</style>
<col">

<header class="print-hidden fixed top-0 left-0 w-full md:left-64 md:w-[calc(100%-16rem)] h-16 bg-blue-900 text-white flex items-center justify-between p-4 shadow-lg z-50">
    <button id="menu-toggle" class="md:hidden text-2xl"><i class="fas fa-bars"></i></button>
    <span class="font-bold text-lg sm:text-xl">
        Lançamento de Notas 
        <?php if ($is_simulating): ?>
            <span class="bg-yellow-500 text-black text-xs px-2 py-1 rounded ml-2">SIMULAÇÃO GESTOR (EDICAO)</span>
        <?php endif; ?>
    </span>
    <div class="flex items-center gap-4 ml-auto">
        <span class="hidden sm:block font-bold"><?= htmlspecialchars($nome_professor) ?></span>
    </div>
</header>

<div id="overlay" class="overlay"></div>

<nav id="sidebar" class="sidebar fixed top-0 left-0 w-64 h-full bg-blue-900 text-white p-4 pt-16 shadow-lg overflow-y-auto">
    <div class="flex flex-col items-center justify-center p-4">
        <span class="text-xl font-bold text-center"><?= htmlspecialchars($nome_professor) ?></span>
    </div>
    <ul class="mt-8 space-y-2">
        <li><a href="painel_professor1.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700"><i class="fas fa-arrow-left"></i> <span>Voltar</span></a></li>
        <li><a href="caderneta_professor.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700"><i class="fas fa-book"></i> <span>Caderneta (AVs)</span></a></li>
        <li><a href="mini_pauta_professor1.php" class="flex items-center gap-3 p-3 rounded-lg bg-blue-700"><i class="fas fa-pen"></i> <span>Mini-Pauta (MT)</span></a></li>
        
        <?php if ($is_simulating): ?>
            <li><a href="retornar_acesso.php" class="flex items-center gap-3 p-3 rounded-lg bg-red-600 hover:bg-red-700 font-bold"><i class="fas fa-undo"></i> <span>Voltar ao Perfil Gestor</span></a></li>
        <?php else: ?>
         ="login.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700"><i class="fas fa-sign-out-alt"></i> <span>Sair</span></a></li>
        <?php endif; ?>
        
    </ul>
</nav>

<main class="content flex-grow p-4 md:p-6 md:ml-64 mt-16">
    <h1 class="text-3xl font-bold text-gray-800 mb-6 print-hidden">Mini-Pauta: Lançamento de NPP e NPT</h1>

    <?php if ($mensagem_sucesso): ?>
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4 print-hidden"><?= htmlspecialchars($mensagem_sucesso) ?></div>
    <?php endif; ?>
    <?php if ($mensagem_erro): ?>
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4 print-hidden"><?= htmlspecialchars($mensagem_erro) ?></div>
    <?php endif; ?>

    <form action="" method="GET" class="form-filtros bg-white p-6 rounded-xl shadow-md mb-6 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 items-end print-hidden">
        <div>
            <label for="turma" class="block text-gray-700 font-semibold mb-2">Turma:</label>
            <select id="turma" name="turma_id" class="w-full p-3 rounded-lg border border-gray-300 focus:ring focus:ring-blue-200 transition-colors" required onchange="this.form.submit()">
                <option value="">Selecione</option>
                <?php foreach ($turmas as $turma): ?>
                    <option value="<?= $turma['id'] ?>" <?= $turma_id == $turma['id'] ? 'selected' : '' ?>><?= $turma['descricao_formatada'] ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div>
            <label for="disciplina" class="block text-gray-700 font-semibold mb-2">Disciplina:</label>
            <select id="disciplina" name="disciplina_id" class="w-full p-3 rounded-lg border border-gray-300 focus:ring focus:ring-blue-200 transition-colors" required>
                <option value="">Selecione</option>
                <?php foreach ($disciplinas as $disciplina): ?>
                    <option value="<?= $disciplina['id'] ?>" <?= $disciplina_id == $disciplina['id'] ? 'selected' : '' ?>><?= $disciplina['nome'] ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div>
            <label for="trimestre" class="block text-gray-700 font-semibold mb-2">Trimestre:</label>
            <select id="trimestre" name="trimestre" class="w-full p-3 rounded-lg border border-gray-300 focus:ring focus:ring-blue-200 transition-colors" required>
                <option value="">Selecione</option>
                <option value="1" <?= $trimestre_selecionado == 1 ? 'selected' : '' ?>>1º</option>
                <option value="2" <?= $trimestre_selecionado == 2 ? 'selected' : '' ?>>2º</option>
                <option value="3" <?= $trimestre_selecionado == 3 ? 'selected' : '' ?>>3º</option>
            </select>
        </div>
        <div class="col-span-1 sm:col-span-2 lg:col-span-1">
            <button type="submit" class="w-full bg-blue-900 text-white py-3 px-6 rounded-lg hover:bg-blue-800 transition"><i class="fas fa-search"></i> Filtrar</button>
        </div>
    </form>

    <?php if ($turma_id && $disciplina_id && $trimestre_selecionado): // INÍCIO DO BLOCO CONDICIONAL PRINCIPAL ?>
        
        <div class="print-hidden mb-4 flex justify-end gap-3">
            
            <?php if ($is_professor || $is_simulating): ?>
                <a href="mini_pauta_anual.php?turma_id=<?= htmlspecialchars($turma_id) ?>&disciplina_id=<?= htmlspecialchars($disciplina_id) ?>" 
                    target="_blank" 
                    class="bg-purple-600 text-white py-2 px-4 rounded-lg hover:bg-purple-700 transition flex items-center gap-2 text-sm font-semibold">
                    <i class="fas fa-file-alt"></i> Caderneta Anual (A4)
                </a>
            <?php endif; ?>
            
            <button onclick="window.print()" class="bg-indigo-600 text-white py-2 px-4 rounded-lg hover:bg-indigo-700 transition flex items-center gap-2 text-sm font-semibold">
                <i class="fas fa-print"></i> Imprimir Mini-Pauta Trimestral
            </button>
        </div>
        <div class="print-table-container bg-white p-6 rounded-xl shadow-md">
            
             <div class="print-header hidden">
                <div class="header-insignia">
                    <img src="insignia/insignia.png" alt="Emblema da República">
                </div>
                
                <div class="header-text">
                    <p style="font-weight: bold; font-size: 11pt;"><?= htmlspecialchars($nome_republica) ?></p>
                    <p style="font-size: 10pt;"><?= htmlspecialchars($nome_governo) ?></p>
                    <p style="font-size: 10pt;"><?= htmlspecialchars($nome_gabinete) ?></p>
                    <p style="font-weight: bold; font-size: 11pt;"><?= htmlspecialchars($nome_escola) ?></p>
                </div>

                <p class="header-title">MINI-PAUTA DO <?= htmlspecialchars($trimestre_selecionado) ?>º TRIMESTRE</p>

                <div class="header-info" style="max-width: 90%; margin: 0 auto;">
                    <p>
                        <span class="info-label">Disciplina:</span>
                        <span class="info-value"><?= htmlspecialchars($nome_disciplina) ?></span>
                    </p>
                    <p>
                        <span class="info-label">Classe:</span>
                        <span class="info-value"><?= htmlspecialchars($classe_dinamica) ?></span>
                        <span class="info-label" style="padding-left: 15px;">Área:</span>
                        <span class="info-value"><?= htmlspecialchars($curso_area_dinamica) ?></span>
                    </p>
                    <p>
                        <span class="info-label">Turma:</span>
                        <span class="info-value"><?= htmlspecialchars($nome_turma_dinamica) ?></span>
                        <span class="info-label" style="padding-left: 15px;">Sala:</span>
                        <span class="info-value"><?= htmlspecialchars($sala_dinamica) ?></span>
                        <span class="info-label" style="padding-left: 15px;">Turno:</span>
                        <span class="info-value"><?= htmlspecialchars($turno_dinamico) ?></span>
                    </p>
                    <p>
                        <span class="info-label">Professor:</span>
                        <span class="info-value"><?= htmlspecialchars($nome_professor) ?></span>
                        <span class="info-label" style="padding-left: 15px;">Ano Lectivo:</span>
                        <span class="info-value"><?= htmlspecialchars($ano_lectivo_atual) ?></span>
                    </p>
                </div>
                <hr style="border: 0; border-top: 1px solid #000; margin: 10px 0;">
            </div>
            <h2 class="text-xl font-bold text-gray-700 mb-4 print-hidden"><?= htmlspecialchars($nome_disciplina) ?> - <?= htmlspecialchars($nome_turma_selecionada) ?> (<?= $trimestre_selecionado ?>º Trimestre)</h2>
<?php
// ---------------------------------------------------------
// BLOQUEIO DE LANÇAMENTO PARA NPP E NPT
// ---------------------------------------------------------

$ano_lectivo_atual = "2025/2026"; // Mantenha isso como está no seu código original
$data_hoje = date("Y-m-d");

// Variáveis de Estado (Inicia como liberado)
$npt_bloqueado = false;
$npp_bloqueado = false;
$motivo_bloqueio_npt = "";
$motivo_bloqueio_npp = "";

// 1. VERIFICAR PRAZO DO NPT
// -------------------------
$tipo_nota = 'NPT';
$stmt_npt = $conn->prepare("
    SELECT data_inicio, data_fim
    FROM prazos_lancamento
    WHERE ano_lectivo = ? AND trimestre = ? AND tipo_nota = ?
    LIMIT 1
");
$stmt_npt->bind_param("sis", $ano_lectivo_atual, $trimestre_selecionado, $tipo_nota);
$stmt_npt->execute();
$resultado_npt = $stmt_npt->get_result()->fetch_assoc();

if ($resultado_npt) {
    // Se encontrou o prazo, atribui as variáveis
    $inicio = $resultado_npt['data_inicio'];
    $fim = $resultado_npt['data_fim'];

    // Verifica se a data atual está fora do prazo
    if ($data_hoje < $inicio) {
        $npt_bloqueado = true;
        $motivo_bloqueio_npt = "O prazo para NPT só inicia em " . date("d/m/Y", strtotime($inicio));
    } elseif ($data_hoje > $fim) {
        $npt_bloqueado = true;
        $motivo_bloqueio_npt = "O prazo para NPT terminou no dia " . date("d/m/Y", strtotime($fim));
    }
    // Não precisa de } aqui, o código está dentro do bloco if($resultado_npt)
} else {
    // Se não encontrou o prazo no banco de dados
    $npt_bloqueado = true;
    $motivo_bloqueio_npt = "O prazo para NPT ainda não foi definido.";
}
$stmt_npt->close();


// 2. VERIFICAR PRAZO DO NPP
// -------------------------
$tipo_nota = 'NPP';
$stmt_npp = $conn->prepare("
    SELECT data_inicio, data_fim
    FROM prazos_lancamento
    WHERE ano_lectivo = ? AND trimestre = ? AND tipo_nota = ?
    LIMIT 1
");
$stmt_npp->bind_param("sis", $ano_lectivo_atual, $trimestre_selecionado, $tipo_nota);
$stmt_npp->execute();
$resultado_npp = $stmt_npp->get_result()->fetch_assoc();

if ($resultado_npp) {
    $inicio = $resultado_npp['data_inicio'];
    $fim = $resultado_npp['data_fim'];

    if ($data_hoje < $inicio) {
        $npp_bloqueado = true;
        $motivo_bloqueio_npp = "O prazo para NPP só inicia em " . date("d/m/Y", strtotime($inicio));
    } elseif ($data_hoje > $fim) {
        $npp_bloqueado = true;
        $motivo_bloqueio_npp = "O prazo para NPP terminou no dia " . date("d/m/Y", strtotime($fim));
    }
} else {
    $npp_bloqueado = true;
    $motivo_bloqueio_npp = "O prazo para NPP ainda não foi definido.";
}
$stmt_npp->close();


// AVISOS DE BLOQUEIO
if ($npt_bloqueado && !$is_simulating): ?>
    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4 print-hidden">
        **NPT Bloqueado:** <?= htmlspecialchars($motivo_bloqueio_npt) ?>
    </div>
<?php endif; ?>
<?php if ($npp_bloqueado && !$is_simulating): ?>
    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4 print-hidden">
        **NPP Bloqueado:** <?= htmlspecialchars($motivo_bloqueio_npp) ?>
    </div>
<?php endif; 
// FIM DOS AVISOS DE BLOQUEIO
?>

<form method="POST" action="salvar_notas.php">
    <?php if ($is_simulating): ?>
        <p class="text-yellow-700 font-semibold mb-4 print-hidden">
            AVISO: O Subdirector Pedagógico está em modo de edição e pode salvar alterações.
        </p>
    <?php endif; ?>
        
    <input type="hidden" name="salvar_notas" value="1">
    <input type="hidden" name="turma_id" value="<?= $turma_id ?>">
    <input type="hidden" name="disciplina_id" value="<?= $disciplina_id ?>">
    <input type="hidden" name="trimestre" value="<?= $trimestre_selecionado ?>">

    <?php
    // Funções auxiliares
    $determina_classe = function($nota) {
        if ($nota === null || $nota === '') return '';
        $nota_float = (float) str_replace(',', '.', $nota);
        return ($nota_float >= 10.0) ? 'nota-aprovada' : 'nota-reprovada';
    };

    $is_nota_vazia = function($nota) {
        return $nota === null || $nota === '' || (float) str_replace(',', '.', $nota) === 0.0;
    };
    ?>
<div class="overflow-x-auto">
        <table class="w-full text-sm border border-gray-300">
            <thead class="bg-blue-900 text-white">
                <tr>
                    <th rowspan="2" class="coluna-nome-aluno-header">Nº</th>
                    <th rowspan="2" class="coluna-nome-aluno-header coluna-nome-aluno-tela">Nome Completo do Aluno</th>
                    <th colspan="4">Trimestre <?= htmlspecialchars($trimestre_selecionado) ?></th>
                    <?php if ($trimestre_selecionado == 3): ?>
                        <th colspan="6">Notas Finais</th>
                    <?php endif; ?>
                    <th rowspan="2">OBS.</th>
                </tr>
                <tr>
                    <th>MAC</th>
                    <th>NPP</th>
                    <th>NPT</th>
                    <th>MT</th>
                    <?php if ($trimestre_selecionado == 3): ?>
                        <th>MFD</th>
                        <th>NEO</th>
                        <th>NEE</th>
                        <th>MEC</th>
                        <th>MF</th>
                        <th>ESTADO</th>
                    <?php endif; ?>
                </tr>
            </thead>
            <tbody>
                <?php $contador = 1; ?>
<?php foreach ($alunos as $aluno): 
    $aluno_id = $aluno['id'];

    // Coleta das notas
    $nota_atual = $notas_por_aluno[$aluno_id][$trimestre_selecionado] ?? [];
    $mac_salvo = $nota_atual['mac'] ?? null; 
    $npp_salvo = $nota_atual['npp'] ?? null;
    $npt_salvo = $nota_atual['npt'] ?? null;
    $mt_salvo  = $nota_atual['mt'] ?? null;
    $obs_salvo = $nota_atual['observacao'] ?? '';
    
    // Colunas NEO/NEE/MFD/MEC/MF/Estado para o 3º Trimestre
    $neo_salvo = $nota_atual['neo'] ?? null;
    $nee_salvo = $nota_atual['nee'] ?? null;

    // --- CÁLCULO MT NO PHP (PARA VALOR INICIAL) ---
    // O MAC é convertido aqui para float para o cálculo
    $mac_float = $mac_salvo !== null ? (float)str_replace(',', '.', $mac_salvo) : null;
    $npp_float = $npp_salvo !== null ? (float)str_replace(',', '.', $npp_salvo) : null;
    $npt_float = $npt_salvo !== null ? (float)str_replace(',', '.', $npt_salvo) : null;
    $mt_calculado = null;

    // FÓRMULA ATUALIZADA: MT = (MAC + NPP + NPT) / 3
    if ($mac_float !== null && $npp_float !== null && $npt_float !== null) {
        $mt_calculado = ($mac_float + $npp_float + $npt_float) / 3;
        $mt_calculado = round($mt_calculado, 2); 
    }

    // Garante que o MT exibido (se não salvo no BD) use o calculado
    $mt_salvo = $mt_salvo ?? $mt_calculado; 
    
    // Classes de cor
    $mac_classe = $determina_classe($mac_salvo);
    $npp_classe = $determina_classe($npp_salvo);
    $npt_classe = $determina_classe($npt_salvo);
    $mt_classe  = $determina_classe($mt_salvo);

    // Cálculo de notas finais (3º trimestre) - Mantido inalterado
    $mfd = $mec = $mf = null;
    $estado_final = '---';
    $mf_display = '---';
    $estado_classe = 'estado-final-incompleto';

    if ($trimestre_selecionado == 3) {
        $mts_calc = [];
        for ($i=1; $i<=3; $i++) {
            $mt_nota = $notas_por_aluno[$aluno_id][$i]['mt'] ?? null;
            // Usa o $mt_salvo atual se for o 3º trimestre, senão usa o salvo no BD
            $mts_calc[$i] = ($i==3 && $mt_salvo!==null) ? (float)str_replace(',', '.', $mt_salvo) : ($mt_nota!==null?(float)str_replace(',', '.', $mt_nota):null);
        }

        $nota_t1 = $mts_calc[1];
        $nota_t2 = $mts_calc[2];
        $nota_t3 = $mts_calc[3];

        $neo_float = $neo_salvo !== null ? (float)str_replace(',', '.', $neo_salvo) : null;
        $nee_float = $nee_salvo !== null ? (float)str_replace(',', '.', $nee_salvo) : null;

        // MFD (Média Final da Disciplina)
        if ($nota_t1!==null && $nota_t2!==null && $nota_t3!==null) $mfd = round(($nota_t1+$nota_t2+$nota_t3)/3,2);
        
        // MEC (Média do Exame de Recurso)
        if ($neo_float!==null && $nee_float!==null && $neo_float>0 && $nee_float>0) $mec = round(($neo_float+$nee_float)/2,2);

        // MF (Média Final)
        if ($mfd!==null) {
            // Lógica de cálculo de MF completa
            if ($mec!==null && $mfd<10) $mf_final = round(($mfd*0.6)+($mec*0.4),2);
            elseif ($nee_float!==null && $nee_float>0 && $mfd<10) $mf_final = round(($mfd*0.6)+($nee_float*0.4),2);
            else $mf_final = round($mfd,2); // Caso MFD puro
            
            // Lógica de arredondamento final e estado
            if ($mf_final>=9.5) { // Arredonda 9.5 para 10
                $mf_final_arredondado = $mf_final >= 9.5 && $mf_final < 10 ? 10 : round($mf_final);
                
                if ($mf_final_arredondado >= 10) {
                    $estado_final='APROVADO';
                    $estado_classe='estado-final-aprovado';
                    $mf_display = number_format($mf_final_arredondado,0,',','.');
                } else {
                    $estado_final='REPROVADO';
                    $estado_classe='estado-final-reprovado';
                    $mf_display = number_format($mf_final,2,',','.');
                }
            } else {
                $estado_final='REPROVADO';
                $estado_classe='estado-final-reprovado';
                $mf_display = number_format($mf_final,2,',','.');
            }
        }

        $mfd_classe = $determina_classe($mfd);
        $neo_classe = $determina_classe($neo_salvo);
        $nee_classe = $determina_classe($nee_salvo);
        $mec_classe = $determina_classe($mec);
        $mf_classe = $determina_classe($mf);
    }
?>
                <tr data-aluno-id="<?= $aluno_id ?>" class="hover:bg-gray-100">
                    <td><?= $contador++ ?></td>
                    <td class="coluna-nome-aluno-tela">
                        <?= htmlspecialchars($aluno['nome_completo'] ?? $aluno['nome']) ?>
                        <input type="hidden" name="aluno_id[]" value="<?= $aluno_id ?>">
                    </td>

                    <td>
                        <input type="text" 
                            name="mac_<?= $aluno_id ?>" 
                            value="<?= $mac_salvo !== null ? str_replace('.', ',', $mac_salvo) : '' ?>" 
                            class="nota-input text-center <?= $mac_classe ?> mac-input" 
                            readonly
                            data-aluno-id="<?= $aluno_id ?>"
                            data-mac-valor="<?= $mac_float ?? '' ?>" >
                    </td>

                    <td>
                        <input type="text" 
                            name="notas[<?= $aluno_id ?>][npp]" 
                            value="<?= $npp_salvo !== null ? str_replace('.', ',', $npp_salvo) : '' ?>" 
                            class="nota-input text-center <?= $npp_classe ?> nota-npp-input" 
                            data-aluno-id="<?= $aluno_id ?>"
                            data-trimestre="<?= $trimestre_selecionado ?>"
                            onkeypress="return onlyNumbersAndComma(event)"
                            <?= $npp_bloqueado && !$is_simulating ? 'readonly' : '' ?>
                            <?= $npp_bloqueado && !$is_simulating ? 'title="'.$motivo_bloqueio_npp.'"' : '' ?>>
                    </td>

                    <td>
                        <input type="text" 
                            name="notas[<?= $aluno_id ?>][npt]" 
                            value="<?= $npt_salvo !== null ? str_replace('.', ',', $npt_salvo) : '' ?>" 
                            class="nota-input text-center <?= $npt_classe ?> nota-npt-input" 
                            data-aluno-id="<?= $aluno_id ?>"
                            data-trimestre="<?= $trimestre_selecionado ?>"
                            onkeypress="return onlyNumbersAndComma(event)"
                            <?= $npt_bloqueado && !$is_simulating ? 'readonly' : '' ?>
                            <?= $npt_bloqueado && !$is_simulating ? 'title="'.$motivo_bloqueio_npt.'"' : '' ?>>
                    </td>

                    <td>
                        <span data-nota-valor="<?= $mt_salvo!==null?number_format($mt_salvo,2,',','.'):'---' ?>" class="print-value nota-calculada <?= $mt_classe ?>">
                            <input type="text" 
                                name="notas[<?= $aluno_id ?>][mt]"
                                id="mt_display_<?= $aluno_id ?>"
                                value="<?= $mt_salvo !== null ? str_replace('.', ',', $mt_salvo) : '' ?>" 
                                class="nota-input text-center <?= $mt_classe ?> mt-display" 
                                readonly style="background:#eee;">
                        </span>
                    </td>

                    <?php if ($trimestre_selecionado==3): ?>
                        <td id="mfd_<?= $aluno_id ?>" 
                            data-nota-valor="<?= $mfd!==null?number_format($mfd,2,',','.'):'---' ?>" 
                            class="print-value nota-calculada <?= $mfd_classe ?>">
                            <?= $mfd!==null?number_format($mfd,2,',','.'):'---' ?>
                        </td>
                        
                        <td>
                            <input type="text" 
                                name="notas[<?= $aluno_id ?>][neo]" 
                                id="neo_<?= $aluno_id ?>" 
                                value="<?= $neo_salvo!==null?str_replace('.', ',', $neo_salvo) : '' ?>" 
                                class="nota-input text-center <?= $neo_classe ?> neo-input" 
                                maxlength="5" placeholder="0,00" 
                                data-aluno-id="<?= $aluno_id ?>"
                                onkeypress="return onlyNumbersAndComma(event)">
                        </td>
                        
                        <td>
                            <input type="text" 
                                name="notas[<?= $aluno_id ?>][nee]" 
                                id="nee_<?= $aluno_id ?>" 
                                value="<?= $nee_salvo!==null?str_replace('.', ',', $nee_salvo) : '' ?>" 
                                class="nota-input text-center <?= $nee_classe ?> nee-input" 
                                maxlength="5" placeholder="0,00" 
                                data-aluno-id="<?= $aluno_id ?>"
                                onkeypress="return onlyNumbersAndComma(event)">
                        </td>
                        
                        <td id="mec_<?= $aluno_id ?>" 
                            data-nota-valor="<?= $mec!==null?number_format($mec,2,',','.'):'---' ?>" 
                            class="print-value nota-calculada <?= $mec_classe ?>">
                            <?= $mec!==null?number_format($mec,2,',','.'):'---' ?>
                        </td>
                        <td id="mf_<?= $aluno_id ?>" 
                            data-nota-valor="<?= $mf_display ?>" 
                            class="print-value nota-final-calc <?= $mf_classe ?>">
                            <?= $mf_display ?>
                        </td>
                        <td id="estado_<?= $aluno_id ?>" 
                            data-nota-valor="<?= $estado_final ?>" 
                            class="print-value <?= $estado_classe ?>">
                            <?= $estado_final ?>
                        </td>
                    <?php endif; ?>

                    <td>
                        <input type="text" 
                            name="notas[<?= $aluno_id ?>][observacao]" 
                            value="<?= htmlspecialchars($obs_salvo) ?>" 
                            class="text-center w-full nota-input" 
                            maxlength="50">
                    </td>
                </tr>
<?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <div class="mt-6 flex justify-end print-hidden">
        <button type="submit" class="bg-green-600 text-white py-3 px-6 rounded-lg hover:bg-green-700 transition"
            <?= ($npp_bloqueado && $npt_bloqueado && !$is_simulating) ? 'disabled' : '' ?>>
            <i class="fas fa-save"></i> Salvar Notas
        </button>
    </div>

</form>
            <div class="print-footer hidden">
                <div class="data-catumbela">
                    <?= htmlspecialchars($localidade) ?>, aos <span><?= date('d') ?></span> de <span><?= strftime('%B') ?></span> de <span><?= date('Y') ?></span>
                </div>
                <div class="footer-grid mt-6">
                    <div>
                        <div class="assinatura-linha"></div>
                        <p class="nome-assinado"><?= htmlspecialchars($nome_professor) ?></p>
                        <p>O Professor(a)</p>
                    </div>
                    <div>
                        <div class="assinatura-linha"></div>
                        <p class="nome-assinado"><?= htmlspecialchars($nome_sub_diretor) ?></p>
                        <p>O Subdirector Pedagógico</p>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; // FIM DO BLOCO CONDICIONAL PRINCIPAL ?>
</main>

<script>
    // ====================================================================
    // Variáveis Globais e Helpers
    // ====================================================================
    
    // Contém as notas MTs dos trimestres anteriores (e o atual ao carregar) para o cálculo do MFD.
    const NOTAS_TRIMESTRAIS_ANTERIORES = <?= $notas_js ?>;
    const TRIMESTRE_ATUAL = <?= $trimestre_js ?>;

    /**
     * Helper para formatar números para exibição (ex: 10.5 -> "10,50" ou "11" se for inteiro).
     */
    function formatarNota(num, decimalPlaces = 2) {
        if (num === null || isNaN(num) || num === undefined) return '---';
        
        // Verifica se é um número inteiro, exceto se for o MFD/MEC/MF (que precisa de 2 casas, a menos que MF seja aprovado)
        if (decimalPlaces === 0) return num.toFixed(0); 
        
        // Arredondamento para aprovação (>= 9.5 vira 10)
        if (num >= 9.5 && decimalPlaces === 2) {
             // Esta lógica deve ser controlada pela função de cálculo MF para o estado final.
             // Aqui, apenas formatamos com 2 casas para entradas/cálculos intermédios.
        }
        
        // Retorna com o número de casas decimais especificado
        return num.toFixed(decimalPlaces).replace('.', ',');
    }

    /**
     * Helper para converter formato de entrada para float (ex: "10,50" -> 10.5).
     */
    function parseNota(str) {
        if (str === null || str === undefined || (typeof str === 'string' && (str.trim() === '' || str.trim() === '---'))) return null;
        
        let parsed = parseFloat(String(str).replace(',', '.'));
        if (isNaN(parsed)) return null;
        
        // Limita a nota entre 0 e 20
        return Math.max(0, Math.min(20, parsed));
    }

    /**
     * Aplica as classes de cor na célula da nota.
     */
    function aplicarClasseCor(element, nota) {
        // Tenta encontrar o input dentro da célula se for o caso
        const target = element.classList.contains('nota-input') ? element : element.querySelector('.nota-input') || element;

        target.classList.remove('nota-aprovada', 'nota-reprovada');

        if (nota !== null && nota !== undefined && !isNaN(nota) && nota > 0) {
            if (nota >= 10.0) {
                target.classList.add('nota-aprovada');
            } else {
                target.classList.add('nota-reprovada');
            }
        }
    }
    
    // ====================================================================
    // Funções de Cálculo Principal (MT e Finais)
    // ====================================================================

    /**
     * Calcula a Média Trimestral (MT) e, no 3º Trimestre, o MFD e chama o cálculo final.
     * FÓRMULA MT: MT = (MAC + NPP + NPT) / 3
     */
    function calculaNotasTrimestrais(inputChanged) {
        const row = inputChanged.closest('tr');
        const alunoId = inputChanged.getAttribute('data-aluno-id');

        // Elementos de nota
        const macInput = row.querySelector('.mac-input');
        const nppInput = row.querySelector('.nota-npp-input');
        const nptInput = row.querySelector('.nota-npt-input');
        const mtDisplayInput = row.querySelector('.mt-display');
        
        // 1. Puxa e normaliza as notas (float ou null)
        // MAC é OBRIGATORIAMENTE puxado do atributo data-mac-valor (carregado do BD)
        const mac = parseNota(macInput.getAttribute('data-mac-valor'));
        const npp = parseNota(nppInput.value);
        const npt = parseNota(nptInput.value);
        
        // 2. Atualiza a cor dos inputs NPP e NPT
        aplicarClasseCor(nppInput, npp);
        aplicarClasseCor(nptInput, npt);

        let mt = null;
        
        // Se todas as notas obrigatórias (MAC, NPP, NPT) estiverem disponíveis, calcula a MT
        if (mac !== null && npp !== null && npt !== null) {
            mt = (mac + npp + npt) / 3;
            mt = Math.round(mt * 100) / 100; // Arredonda para 2 casas
        }

        // 3. Atualiza o valor na tela (input e display span do MT)
        if (mtDisplayInput) {
            const mtCell = mtDisplayInput.closest('td').querySelector('.print-value');
            
            // Atualiza o INPUT READONLY (valor que será enviado no POST)
            mtDisplayInput.value = mt !== null ? formatarNota(mt).replace('.', ',') : '';
            aplicarClasseCor(mtDisplayInput, mt);

            // Atualiza o SPAN para exibição e impressão
            if (mtCell) {
                mtCell.textContent = formatarNota(mt);
                mtCell.setAttribute('data-nota-valor', formatarNota(mt));
                aplicarClasseCor(mtCell, mt);
            }
        }

        // 4. Se for o 3º Trimestre, calcula o MFD e chama o cálculo final
        if (TRIMESTRE_ATUAL == 3) {
            let mfd = null;
            
            // Obter MT1 e MT2 salvos (garantindo que se o valor for nulo/vazio, seja 0 para o cálculo)
            const mt1 = parseNota(NOTAS_TRIMESTRAIS_ANTERIORES[alunoId] && NOTAS_TRIMESTRAIS_ANTERIORES[alunoId][1] ? NOTAS_TRIMESTRAIS_ANTERIORES[alunoId][1].mt : null) || 0;
            const mt2 = parseNota(NOTAS_TRIMESTRAIS_ANTERIORES[alunoId] && NOTAS_TRIMESTRAIS_ANTERIORES[alunoId][2] ? NOTAS_TRIMESTRAIS_ANTERIORES[alunoId][2].mt : null) || 0;
            const mt3 = mt !== null ? mt : 0; // Usa o MT3 recém-calculado (ou 0 se incompleto)

            // Calcula MFD se todos os 3 MTs estiverem disponíveis (MT > 0.0)
            if (mt1 > 0 && mt2 > 0 && mt3 > 0) {
                mfd = (mt1 + mt2 + mt3) / 3;
                mfd = Math.round(mfd * 100) / 100;
            }
            
            // 5. Atualiza MFD na tela
            const mfdCell = document.getElementById(`mfd_${alunoId}`);
